#include "godot.h"

/*
 * declare the exported functions
 */
__declspec(dllexport) p_godot_device_spec* godot_get_supported_devices( void );
__declspec(dllexport) p_godot_device_spec  godot_get_device_spec( char* name, GODOT_UINT64 version );

/*****************************************************************************************
    vLoopCtrl

	clk		One-bit input
	err_in		Six-bit signed input
	out		14-bit unsigned output

    This device models the transfer function of the analog amplifier in the voltage-loop
	control of a closed-loop regulated PFC in one of our example directories.  The
	analog transfer function has two poles and one zero.  This transfer function is
	implemented through a digital filter here.
*****************************************************************************************/

/*
 * vLoopCtrl function declarations
 */
void
    vLoopCtrl_set_initial_condition( p_godot_device device );

void
    vLoopCtrl_action( p_godot_device device );


/*
 * vLoopCtrl utility struct
 */
typedef struct t_vLoopCtrl_vars
{
    GODOT_DOUBLE	delay;

    GODOT_INT64		i64_tmp_max;
    GODOT_INT64		out_d1;
    GODOT_INT64		out_d0;

    GODOT_INT32		i_scaler;

    GODOT_UINT16	num_outBits;

    GODOT_BYTE8		err_in_n1;
    GODOT_BYTE8		err_in_n0;

} s_vLoopCtrl_vars, *p_vLoopCtrl_vars;


/*
 * vLoopCtrl parameter specifications
 */
const static GODOT_PARAMETER_SPEC(vLoopCtrl, IC_in_volts, GODOT_PARAMETER_TYPE_DOUBLE, 2.36);
const static GODOT_PARAMETER_SPEC(vLoopCtrl, max_out_in_volts, GODOT_PARAMETER_TYPE_DOUBLE, 13.0);

const static GODOT_PARAMETER_SPEC(vLoopCtrl, N2, GODOT_PARAMETER_TYPE_INTEGER, 2451);
const static GODOT_PARAMETER_SPEC(vLoopCtrl, N1, GODOT_PARAMETER_TYPE_INTEGER, 2);
const static GODOT_PARAMETER_SPEC(vLoopCtrl, N0, GODOT_PARAMETER_TYPE_INTEGER, -2449);

const static GODOT_PARAMETER_SPEC(vLoopCtrl, D2, GODOT_PARAMETER_TYPE_INTEGER, 65536);
const static GODOT_PARAMETER_SPEC(vLoopCtrl, D1, GODOT_PARAMETER_TYPE_INTEGER, -130801);
const static GODOT_PARAMETER_SPEC(vLoopCtrl, D0, GODOT_PARAMETER_TYPE_INTEGER, 65265);

const static GODOT_PARAMETER_SPEC(vLoopCtrl, A2D_gain, GODOT_PARAMETER_TYPE_DOUBLE, 500.0);

static const s_godot_parameter_spec *vLoopCtrl_parameters[] =
    {
	&( GODOT_PARAMETER_SPEC_NAME( vLoopCtrl, IC_in_volts ) )
	,
	&( GODOT_PARAMETER_SPEC_NAME( vLoopCtrl, max_out_in_volts ) )
	,
	&( GODOT_PARAMETER_SPEC_NAME( vLoopCtrl, N2 ) )
	,
	&( GODOT_PARAMETER_SPEC_NAME( vLoopCtrl, N1 ) )
	,
	&( GODOT_PARAMETER_SPEC_NAME( vLoopCtrl, N0 ) )
	,
	&( GODOT_PARAMETER_SPEC_NAME( vLoopCtrl, D2 ) )
	,
	&( GODOT_PARAMETER_SPEC_NAME( vLoopCtrl, D1 ) )
	,
	&( GODOT_PARAMETER_SPEC_NAME( vLoopCtrl, D0 ) )
	,
	&( GODOT_PARAMETER_SPEC_NAME( vLoopCtrl, A2D_gain ) )
	,
	NULL
    };


/*
 * vLoopCtrl connectivity specifications
 */
const static char*	vLoopCtrl_input_specs[]  = { "clk,1", "err_in,6", NULL };
const static char*	vLoopCtrl_output_specs[] = { "out,14", NULL };


/*
 * vLoopCtrl device specification
 */
static s_godot_device_spec  vLoopCtrl_device_spec =
    {
	.name = "vLoopCtrl"
	,
	.version = 1
	,
	.input_specs = (char**) vLoopCtrl_input_specs
	,
	.output_specs = (char**) vLoopCtrl_output_specs
	,
	.parameter_specs = (s_godot_parameter_spec**) vLoopCtrl_parameters
	,
	.set_initial_condition = vLoopCtrl_set_initial_condition
	,
	.action = vLoopCtrl_action
    };


/*****************************************************************************************
    multiplier

	clk		One-bit input
	AC_in		9-bit unsigned rectified AC voltage
	vLoopCtrl_out	14-bit unsigned output from vLoopCtrl
	out		16-bit unsigned output of (AC_in * vLoopCtrl_out)

    This device is simply a multiplier with a clocked register output.
*****************************************************************************************/

/*
 * multiplier function declarations
 */
void
    multiplier_set_initial_condition( p_godot_device device );

void
    multiplier_action( p_godot_device device );


/*
 * multiplier utility struct
 */
typedef struct t_multiplier_vars
{
    GODOT_DOUBLE	delay;

    GODOT_UINT32	ui32_out_max;			// Maximum value for the output "out"

    GODOT_UINT16	num_outBits;
    GODOT_UINT16	num_AC_in_Bits;			// Number of input bits from the AC_in ADC
    GODOT_UINT16	num_vLoopCtrl_out_Bits;		// Number of input bits from the output of vLoopCtrl
    GODOT_UINT16	num_total_inBits;		// Total number of input bits
    GODOT_UINT16	num_rshiftBits;			// Number of trailing bits to discard from the product

} s_multiplier_vars;


/*
 * multiplier parameter specifications
 */
const static GODOT_PARAMETER_SPEC(multiplier, IC_in_volts, GODOT_PARAMETER_TYPE_DOUBLE, 0.0);
const static GODOT_PARAMETER_SPEC(multiplier, max_out_in_volts, GODOT_PARAMETER_TYPE_DOUBLE, 13.0);

const static GODOT_PARAMETER_SPEC(multiplier, A2D_V_FB_gain, GODOT_PARAMETER_TYPE_DOUBLE, 500.0);
const static GODOT_PARAMETER_SPEC(multiplier, A2D_AC_IN_gain, GODOT_PARAMETER_TYPE_DOUBLE, 500.0);

static const s_godot_parameter_spec *multiplier_parameters[] =
    {
	&( GODOT_PARAMETER_SPEC_NAME( multiplier, IC_in_volts ) )
	,
	&( GODOT_PARAMETER_SPEC_NAME( multiplier, max_out_in_volts ) )
	,
	&( GODOT_PARAMETER_SPEC_NAME( multiplier, A2D_V_FB_gain ) )
	,
	&( GODOT_PARAMETER_SPEC_NAME( multiplier, A2D_AC_IN_gain ) )
	,
	NULL
    };


/*
 * multiplier connectivity specifications
 */
const static char*	multiplier_input_specs[]  = { "clk,1", "AC_in,9", "vLoopCtrl_out,14", NULL };
const static char*	multiplier_output_specs[] = { "out,16", NULL };


/*
 * multiplier device specification
 */
static s_godot_device_spec  multiplier_device_spec =
    {
	.name = "multiplier"
	,
	.version = 1
	,
	.input_specs = (char**) multiplier_input_specs
	,
	.output_specs = (char**) multiplier_output_specs
	,
	.parameter_specs = (s_godot_parameter_spec**) multiplier_parameters
	,
	.set_initial_condition = multiplier_set_initial_condition
	,
	.action = multiplier_action
    };



/*
 * Supported devices
 */
static p_godot_device_spec supported_devices[] =
    {
	&vLoopCtrl_device_spec,
	&multiplier_device_spec,
	NULL
    };

/*
 * function implementations
 */
p_godot_device_spec* godot_get_supported_devices(void)
    {
	return supported_devices;
    }

p_godot_device_spec godot_get_device_spec(char* name, GODOT_UINT64 version)

{
	p_godot_device_spec spec = NULL;

	if ( 0 == _stricmp( "vLoopCtrl", name ) )
	{
	    // if version checking is necessary, add it here...
	    spec = &vLoopCtrl_device_spec;
	}

	else if ( 0 == _stricmp( "multiplier", name ) )
	{
	    // if version checking is necessary, add it here...
	    spec = &multiplier_device_spec;
	}

	return spec;

}

/*
 * vLoopCtrl_set_initial_condition
 */
void vLoopCtrl_set_initial_condition( p_godot_device device )

{

    p_godot_bus		out_bus = NULL;

    p_godot_parameter	IC_param = NULL;
    p_godot_parameter	max_out_param = NULL;
    p_godot_parameter	A2D_gain_param = NULL;

    p_vLoopCtrl_vars	vLoopCtrl_vars_p = NULL;

    s_godot_bus_conversion
			conversion;

    GODOT_UINT16	out_initCond;
    GODOT_UINT32	ui32_out_max;
    GODOT_DOUBLE	dbl_tmp;


    if (    GODOT_NO_ERROR != device->funcs->godot_get_bus_by_name(device, "out", GODOT_DIRECTION_OUTPUT, &out_bus)
		||
	    GODOT_NO_ERROR != device->funcs->godot_get_parameter_by_name(device, "IC_in_volts", &IC_param)
		||
	    GODOT_NO_ERROR != device->funcs->godot_get_parameter_by_name(device, "max_out_in_volts", &max_out_param)
		||
	    GODOT_NO_ERROR != device->funcs->godot_get_parameter_by_name(device, "A2D_gain", &A2D_gain_param)		)

	{
		// TODO: panic
		DebugBreak();
	}

    else
	{
	    // Allocate the user storage - USER SHOULD CHECK RETURN VALUE
	    if ( GODOT_NO_ERROR != device->funcs->godot_instantiate_user_storage(device, sizeof(s_vLoopCtrl_vars)) )
		{
		    // TODO: panic
		    DebugBreak();
		    return;
		}

	    else
		{
		    // Map the user storage
		    vLoopCtrl_vars_p = (s_vLoopCtrl_vars *) device->user_storage;

		    // Assign values
		    vLoopCtrl_vars_p->num_outBits = 14;
		    vLoopCtrl_vars_p->i_scaler = 16384;
		    vLoopCtrl_vars_p->delay    = 2e-12;

		    // Set the value of ui32_out_max according to the number of
		    //   output bits
		    ui32_out_max =   0xFFFFFFFF;
		    ui32_out_max >>= (32 - vLoopCtrl_vars_p->num_outBits);

		    // Reduce ui32_out_max according to max_out_in_volts and A2D_gain.
		    //   No checking if dbl_tmp is less than zero.  Only a fool will
		    //   assign negative value to either max_out_in_volts or A2D_gain.
		    dbl_tmp = max_out_param->value.double_value * A2D_gain_param->value.double_value;
		    if ( dbl_tmp < ui32_out_max )
			ui32_out_max = (GODOT_UINT32) dbl_tmp;

		    // Figure out the initial value of out_initCond according to the
		    //   values of IC_in_volts and A2D_gain.
		    dbl_tmp = IC_param->value.double_value * A2D_gain_param->value.double_value;
		    if ( dbl_tmp < 0.0 )
			dbl_tmp = 0.0;
		     out_initCond = (GODOT_UINT16)( dbl_tmp );

		    // Limit out_initCond according to ui32_out_max
		    if ( out_initCond > ui32_out_max )
			out_initCond = ui32_out_max;

		    // Assign the output bus value according to the initial condition
		    conversion.encoding = GODOT_ENCODING_UNSIGNED;
		    conversion.type     = GODOT_TYPE_16BIT;
		    conversion.uint16   = out_initCond;

		    // write to the bus
		    if ( GODOT_NO_ERROR != device->funcs->godot_write_bus( out_bus, &conversion, GODOT_WRITE_INITIAL_CONDITION ) )
				{
				    // TODO: panic
				    DebugBreak();
				}

		    // Set the value of i64_tmp_max for later use in the action
		    //   function
		    vLoopCtrl_vars_p->i64_tmp_max = ui32_out_max * vLoopCtrl_vars_p->i_scaler;

		    vLoopCtrl_vars_p->err_in_n1 = -1;
		    vLoopCtrl_vars_p->err_in_n0 = -1;

		    vLoopCtrl_vars_p->out_d1 =
		    vLoopCtrl_vars_p->out_d0 = out_initCond * vLoopCtrl_vars_p->i_scaler;
		}

	}

}


/*
 * vLoopCtrl_action
 */
void vLoopCtrl_action( p_godot_device device )

{

    p_godot_bus		out_bus = NULL;
    p_godot_bus		err_in_bus = NULL;

    p_godot_pin		clk_pin = NULL;

    p_godot_parameter	N2_param = NULL;
    p_godot_parameter	N1_param = NULL;
    p_godot_parameter	N0_param = NULL;
    p_godot_parameter	D2_param = NULL;
    p_godot_parameter	D1_param = NULL;
    p_godot_parameter	D0_param = NULL;

    p_vLoopCtrl_vars	vLoopCtrl_vars_p = NULL;

    s_godot_bus_conversion
			conversion;

    GODOT_INT16		err_in;
    GODOT_INT64		i64_tmp;


    if (    GODOT_NO_ERROR != device->funcs->godot_get_bus_by_name(device, "out", GODOT_DIRECTION_OUTPUT, &out_bus)
		||
	    GODOT_NO_ERROR != device->funcs->godot_get_bus_by_name(device, "err_in", GODOT_DIRECTION_INPUT, &err_in_bus)
		||
	    GODOT_NO_ERROR != device->funcs->godot_get_pin_by_bus_and_index(device, "clk", 0, GODOT_DIRECTION_INPUT, &clk_pin)
		||
	    GODOT_NO_ERROR != device->funcs->godot_get_parameter_by_name(device, "N2", &N2_param)
		||
	    GODOT_NO_ERROR != device->funcs->godot_get_parameter_by_name(device, "N1", &N1_param)
		||
	    GODOT_NO_ERROR != device->funcs->godot_get_parameter_by_name(device, "N0", &N0_param)
		||
	    GODOT_NO_ERROR != device->funcs->godot_get_parameter_by_name(device, "D2", &D2_param)
		||
	    GODOT_NO_ERROR != device->funcs->godot_get_parameter_by_name(device, "D1", &D1_param)
		||
	    GODOT_NO_ERROR != device->funcs->godot_get_parameter_by_name(device, "D0", &D0_param)					)

	{
	    // TODO: panic
	    DebugBreak();
	}

    else
	{
	    // Check for a positive clock edge
	    if ( is_GODOT_POSITIVE_EDGE(clk_pin) )
		{
		    // Read the err_in bus
		    conversion.encoding = GODOT_ENCODING_TWOS_COMPLEMENT;
		    conversion.type     = GODOT_TYPE_AUTO;
		    if ( GODOT_NO_ERROR != device->funcs->godot_read_bus(err_in_bus, &conversion) )
				{
				    // TODO: panic
				    DebugBreak();
				    return;
				}

			    else
				{
				    // Assign err_in
				    err_in = (GODOT_INT16) conversion.byte8;
				}

		    // Map the user storage
		    vLoopCtrl_vars_p = (s_vLoopCtrl_vars *) device->user_storage;

		    // Compute i64_tmp, which is a scaled version of the next
		    //   output before any limiting is applied
		    i64_tmp =    N2_param->value.int_value * err_in
			       + N1_param->value.int_value * vLoopCtrl_vars_p->err_in_n1
			       + N0_param->value.int_value * vLoopCtrl_vars_p->err_in_n0;

		    i64_tmp *=  vLoopCtrl_vars_p->i_scaler;

		    i64_tmp -=   D1_param->value.int_value * vLoopCtrl_vars_p->out_d1
			       + D0_param->value.int_value * vLoopCtrl_vars_p->out_d0;
		    i64_tmp /=   D2_param->value.int_value;

		    // Enforce the bounds on i_tmp
		    if ( i64_tmp < 0 )
				{
				    i64_tmp = 0;
				}

			    else if ( i64_tmp > vLoopCtrl_vars_p->i64_tmp_max )
				{
				    i64_tmp = vLoopCtrl_vars_p->i64_tmp_max;
				}

		    // Shift the registers associated with err_in
		    vLoopCtrl_vars_p->err_in_n0 = vLoopCtrl_vars_p->err_in_n1;
		    vLoopCtrl_vars_p->err_in_n1 = err_in;

		    // Shift the registers associated with out
		    vLoopCtrl_vars_p->out_d0 = vLoopCtrl_vars_p->out_d1;
		    vLoopCtrl_vars_p->out_d1 = i64_tmp;

		    // write to "out" bus
		    conversion.encoding = GODOT_ENCODING_UNSIGNED;
		    conversion.type     = GODOT_TYPE_16BIT;
		    conversion.uint16   = (GODOT_INT16)( i64_tmp / vLoopCtrl_vars_p->i_scaler );
		    if ( GODOT_NO_ERROR != device->funcs->godot_write_bus(out_bus, &conversion, vLoopCtrl_vars_p->delay) )
				{
				    // TODO: panic
				    DebugBreak();
				}

		} /* if ( is_GODOT_POSITIVE_EDGE(clk_pin) ) */

	}

}


/*
 * multiplier_set_initial_condition
 */
void multiplier_set_initial_condition( p_godot_device device )

{

    p_godot_bus		out_bus = NULL;

    p_godot_parameter	IC_param = NULL;
    p_godot_parameter	max_out_param = NULL;
    p_godot_parameter	A2D_V_FB_gain_param = NULL;
    p_godot_parameter	A2D_AC_IN_gain_param = NULL;

    s_multiplier_vars	*multiplier_vars_p = NULL;

    s_godot_bus_conversion
			conversion;

    GODOT_UINT16	out_initCond;
    GODOT_INT32		i_scaler;
    GODOT_DOUBLE	dbl_tmp;


    if (    GODOT_NO_ERROR != device->funcs->godot_get_bus_by_name(device, "out", GODOT_DIRECTION_OUTPUT, &out_bus)
		||
	    GODOT_NO_ERROR != device->funcs->godot_get_parameter_by_name(device, "IC_in_volts", &IC_param)
		||
	    GODOT_NO_ERROR != device->funcs->godot_get_parameter_by_name(device, "max_out_in_volts", &max_out_param)
		||
	    GODOT_NO_ERROR != device->funcs->godot_get_parameter_by_name(device, "A2D_V_FB_gain", &A2D_V_FB_gain_param)
		||
	    GODOT_NO_ERROR != device->funcs->godot_get_parameter_by_name(device, "A2D_AC_IN_gain", &A2D_AC_IN_gain_param)	)

	{
	    // TODO: panic
	    DebugBreak();
	}

    else
	{
	    // Allocate the user storage - USER SHOULD CHECK RETURN VALUE
	    if ( GODOT_NO_ERROR != device->funcs->godot_instantiate_user_storage(device, sizeof(s_multiplier_vars)))
		{
		    // TODO: panic
		    DebugBreak();
		}

	    else
		{
		    // Map the user storage
		    multiplier_vars_p = (s_multiplier_vars *) device->user_storage;

		    // Assign values
		    multiplier_vars_p->num_outBits		  = 16;
		    multiplier_vars_p->num_AC_in_Bits	  	= 9;
		    multiplier_vars_p->num_vLoopCtrl_out_Bits	= 14;
		    multiplier_vars_p->delay    		  = 2e-12;

		    multiplier_vars_p->num_total_inBits =   multiplier_vars_p->num_AC_in_Bits
							  + multiplier_vars_p->num_vLoopCtrl_out_Bits;

		    // Set the number of trailng bits to discard from the product
		    //   and i_scaler
		    multiplier_vars_p->num_rshiftBits =   multiplier_vars_p->num_total_inBits
							- multiplier_vars_p->num_outBits;

		    i_scaler = 1 << multiplier_vars_p->num_rshiftBits;

		    // Set the value of ui32_out_max according to the number of
		    //   output bits
		    multiplier_vars_p->ui32_out_max =   0xFFFFFFFF;
		    multiplier_vars_p->ui32_out_max >>= (32 - multiplier_vars_p->num_outBits);

		    // Reduce ui32_out_max according to max_out_in_volts, A2D_V_FB_gain,
		    //   and A2D_AC_IN_gain.  No checking if dbl_tmp is less than zero.
		    dbl_tmp =   max_out_param->value.double_value
			      * A2D_V_FB_gain_param->value.double_value
			      * A2D_AC_IN_gain_param->value.double_value;

		    dbl_tmp /= i_scaler;
		    if ( dbl_tmp < multiplier_vars_p->ui32_out_max )
			multiplier_vars_p->ui32_out_max = (GODOT_UINT32) (dbl_tmp);

		    // Figure out the initial value of out_initCond according to the
		    //   values of IC_in_volts, A2D_V_FB_gain, and A2D_AC_IN_gain.
		    dbl_tmp =   IC_param->value.double_value
			      * A2D_V_FB_gain_param->value.double_value
			      * A2D_AC_IN_gain_param->value.double_value;

		    dbl_tmp /= i_scaler;

		    if ( dbl_tmp < 0.0 )
			dbl_tmp = 0.0;
		     out_initCond = (GODOT_UINT16)( dbl_tmp );

		    // Limit out_initCond according to ui32_out_max
		    if ( out_initCond > multiplier_vars_p->ui32_out_max )
			out_initCond = multiplier_vars_p->ui32_out_max;

		    // Assign the output bus value according to the initial condition
		    conversion.encoding = GODOT_ENCODING_UNSIGNED;
		    conversion.type     = GODOT_TYPE_16BIT;
		    conversion.uint16   = out_initCond;

		    // write to the bus
		    if ( GODOT_NO_ERROR != device->funcs->godot_write_bus(out_bus, &conversion, GODOT_WRITE_INITIAL_CONDITION) )
				{
				    // TODO: panic
				    DebugBreak();
				}
		}

	}

}


/*
 * multiplier_action
 */
void multiplier_action( p_godot_device device )

{

    p_godot_bus		out_bus = NULL;
    p_godot_bus		AC_in_bus = NULL;
    p_godot_bus		vLoopCtrl_out_bus = NULL;

    p_godot_pin		clk_pin = NULL;

    s_multiplier_vars	*multiplier_vars_p = NULL;

    s_godot_bus_conversion
			conversion;

    GODOT_UINT16	AC_in;
    GODOT_UINT16	vLoopCtrl_out;

    GODOT_UINT32	ui32_out;


    if (    GODOT_NO_ERROR != device->funcs->godot_get_bus_by_name(device, "out", GODOT_DIRECTION_OUTPUT, &out_bus)
		||
	    GODOT_NO_ERROR != device->funcs->godot_get_bus_by_name(device, "AC_in", GODOT_DIRECTION_INPUT, &AC_in_bus)
		||
	    GODOT_NO_ERROR != device->funcs->godot_get_bus_by_name(device, "vLoopCtrl_out", GODOT_DIRECTION_INPUT, &vLoopCtrl_out_bus)
		||
	    GODOT_NO_ERROR != device->funcs->godot_get_pin_by_bus_and_index(device, "clk", 0, GODOT_DIRECTION_INPUT, &clk_pin)		)

	{
	    // TODO: panic
	    DebugBreak();
	}

    else
	{
	    // Check for a positive clock edge
	    if ( is_GODOT_POSITIVE_EDGE(clk_pin) )
		{
		    // Read the AC_in bus
		    conversion.encoding = GODOT_ENCODING_UNSIGNED;
		    conversion.type     = GODOT_TYPE_AUTO;
		    if ( GODOT_NO_ERROR != device->funcs->godot_read_bus(AC_in_bus, &conversion) )
				{
				    // TODO: panic
				    DebugBreak();
				    return;
				}

			    else
				{
				    // Assign AC_in
				    AC_in = (GODOT_UINT16) conversion.uint16;
				}


		    // Read the vLoopCtrl_out bus
		    conversion.encoding = GODOT_ENCODING_UNSIGNED;
		    conversion.type     = GODOT_TYPE_AUTO;
		    if ( GODOT_NO_ERROR != device->funcs->godot_read_bus(vLoopCtrl_out_bus, &conversion) )
				{
				    // TODO: panic
				    DebugBreak();
				    return;
				}

			    else
				{
				    // Assign vLoopCtrl_out
				    vLoopCtrl_out = (GODOT_UINT16) conversion.uint16;
				}


		    // Map the user storage
		    multiplier_vars_p = (s_multiplier_vars *) device->user_storage;

		    ui32_out = (vLoopCtrl_out * AC_in) >> multiplier_vars_p->num_rshiftBits;
		    if ( ui32_out > multiplier_vars_p->ui32_out_max )
			ui32_out = multiplier_vars_p->ui32_out_max;

		    // write to "out" bus
		    conversion.encoding = GODOT_ENCODING_UNSIGNED;
		    conversion.type     = GODOT_TYPE_16BIT;
		    conversion.uint16   = (GODOT_INT16)( ui32_out );
		    if ( GODOT_NO_ERROR != device->funcs->godot_write_bus(out_bus, &conversion, multiplier_vars_p->delay) )
				{
				    // TODO: panic
				    DebugBreak();
				}

		} /* if ( is_GODOT_POSITIVE_EDGE(clk_pin) ) */

	}

}
